
#include "config.h"
#include <string.h>
#include <stdio.h>

#include "cli.h"
#include "configuration.h"
#include "max6682.h"
#include "psmonitor.h"


/*

P1.0 TA0 Heat0
P1.2 TA1 Heat1
P1.3 CS0
P1.4 CS1
P1.5 CS2
P1.6 Clock
P1.7 SO
P2.0 TA2 Heat2

*/

#define CKSET() P1OUT|=BV(6)
#define CKCLEAR() P1OUT&=~BV(6)
#define SOREAD() (P1IN&BV(7))


int max6682Init(void)
{
	P1OUT|=7<<3;     /* all CS high  */
	CKCLEAR();
	/* setup timerA to be PWM  */

	TACTL= MC_2 |		/* continuous up  */
		TASSEL_2 |	/* input clock is SMCLK  */
		ID_0 |		/* input clock is divided by 1 */
		// ID_1 |	/* input clock is divided by 2 */
		// TAIE |	/* enable ISR on timer overflow  */
		0
		;

	TACCTL0= OUTMOD_0;	/* out mode  */
	TACCTL1= OUTMOD_6;	/* Set on CCR0,  toggle on CCRx  */
	TACCTL2= OUTMOD_6;

	TACCR0= 0xFFFF;
	TACCR1= 0;
	TACCR2= 0;

	return 0;
}

/* Returns DAC units
*/
int max6682Read(int p)
{
	int v=0;
	int i;

	if ((p<0) || (p>2))
		return -1;

	P1OUT&=~(1<<(p+3));    /* clear the appropriate CS  */
	CKSET();
	if (SOREAD())         /* MSB is sign bit  */
		v=0xFFFF;
	for(i=0;i<10;i++)
	{
		CKCLEAR();
		CKSET();
		v<<=1;
		if (SOREAD())
			v|=1;
	}
	P1OUT|=7<<3;     /* all CS high  */
	CKCLEAR();

	return v;
}

/* DAC value which if exceeded will turn off the 24V power supply
*/
#define FAILSAFE 320

/* These values are from curve-fitting a bunch of samples
*/
#define SLOPE 7.44935065
#define YINT -1973.91515152

double dactokelvin(int d)
{
	return ((double)d-YINT)/SLOPE;
}

int kelvintodac(int k)
{
	return ((double)k*SLOPE)+YINT;
}

int max6682Show(void)
{
	int i,r,k,sk;

	writeStrLong("\r\nTemps\r\nnum setpoint (dac) observed (dac)\r\n");
	for(i=0;i<NUMTEMP;i++)
	{
		r=max6682Read(i);
		k=dactokelvin(r)*10.0;
		sk=dactokelvin(curconfig.tempsetpoint[i])*10.0;
		snprintf(msgbuff,sizeof(msgbuff),"%d    %d (%d)   %d (%d)\r\n",i,sk,curconfig.tempsetpoint[i],k,r);
		writeStrLong(msgbuff);
	}
	return 0;
}

void max6682Set(int chan, int kel)
{
	if ((chan<0) || (chan>(NUMTEMP-1)))
		return;
	curconfig.tempsetpoint[chan]=kelvintodac(kel/10.0);
}

/* This is called at about 1 Hz by main
*/
int max6682Poll(void)
{
	int i,r,p;
	int window=10;
	
	for(i=0;i<NUMTEMP;i++)
	{
		r=max6682Read(i);
		if (r>FAILSAFE)
		{
			psmonitor_ACOff();
			*(&TACCR0+i)=0;
			snprintf(msgbuff,sizeof(msgbuff),"  temp %d %d FAILSAFE\r\n",i,r);
			writeStrLong(msgbuff);
		}
		if ((curconfig.tempsetpoint[i]>0) && (r>0))   // if enabled, and measurement is valid...
		{
			if (r>=curconfig.tempsetpoint[i])
				*(&TACCR0+i)=0;
			else
			{
				p=curconfig.tempsetpoint[i]-r;
				if (p>window)
					*(&TACCR0+i)=0xffff;
				else
				{
					*(&TACCR0+i)=0xffff*(p/(double)window);
				}
			}
#if 0
			snprintf(msgbuff,sizeof(msgbuff),"temp %d %d %u\r\n",i,r,*(&TACCR0+i));
			writeStrLong(msgbuff);
#endif
		}
		else
			*(&TACCR0+i)=0;
	}

	return 0;
}
